<?php
/**
 * This file stores all default settings.
 *
 * @category Admin
 * @package  OxyProps
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @license  https://www.gnu.org/licenses/gpl-2.0.html  GPL v2 or later
 * @link     https://oxyprops.com                       OxyProps Website
 * @since    1.4.0
 */

namespace OxyProps\Inc;

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Stores and returns plugin defaults
 *
 * @author   Cédric Bontems <cedric@thewebforge.dev>
 * @since    1.4.0
 */
final class Defaults {


	/**
	 * Unique instance of Defaults
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @var object Self stored instance
	 */
	private static $instance = null;

	/**
	 * Returns the unique instance of Defaults
	 * and creates one if it does not already exist.
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return object Defaults singleton.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new Defaults();
		}
		return self::$instance;
	}

	/**
	 * Returns the OxyProps version of Oxygen Global Settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array Gloabal settings for Oxygen.
	 */
	public function get_oxyprops_global_settings_for_oxygen() {
		$defaults = array();

		$defaults['fonts'] = array(
			'Text'    => 'var(--o-font-sans)',
			'Display' => 'var(--o-font-sans)',
		);

		$defaults['scripts'] = array(
			'scroll_to_hash'        => '',
			'scroll_to_hash_time'   => '',
			'scroll_to_hash_offset' => '',
		);

		$defaults['indicateParents'] = 'true';

		$defaults['classSuggestionsLimit'] = '15';

		$defaults['headings'] = array(
			'H1' => array(
				'font-size-unit' => ' ',
				'font-size'      => 'var(--o-fluid-font)',
				'font-weight'    => 'var(--o-font-weight-7)',
				'line-height'    => 'var(--o-font-lineheight-1)',
				'color'          => 'var(--o-text-1)',
			),
			'H2' => array(
				'font-size-unit' => ' ',
				'font-size'      => 'var(--o-fluid-font)',
				'font-weight'    => 'var(--o-font-weight-7)',
				'line-height'    => 'var(--o-font-lineheight-1)',
				'color'          => 'var(--o-text-1)',
			),
			'H3' => array(
				'font-size-unit' => ' ',
				'font-size'      => 'var(--o-fluid-font)',
				'font-weight'    => 'var(--o-font-weight-7)',
				'line-height'    => 'var(--o-font-lineheight-1)',
				'color'          => 'var(--o-text-1)',
			),
			'H4' => array(
				'font-size-unit' => ' ',
				'font-size'      => 'var(--o-fluid-font)',
				'font-weight'    => 'var(--o-font-weight-7)',
				'line-height'    => 'var(--o-font-lineheight-1)',
				'color'          => 'var(--o-text-1)',
			),
			'H5' => array(
				'font-size-unit' => ' ',
				'font-size'      => 'var(--o-fluid-font)',
				'font-weight'    => 'var(--o-font-weight-7)',
				'line-height'    => 'var(--o-font-lineheight-1)',
				'color'          => 'var(--o-text-1)',
			),
			'H6' => array(
				'font-size-unit' => ' ',
				'font-size'      => 'var(--o-fluid-font)',
				'font-weight'    => 'var(--o-font-weight-7)',
				'line-height'    => 'var(--o-font-lineheight-1)',
				'color'          => 'var(--o-text-1)',
			),
		);

		$defaults['body_text'] = array(
			'font-size-unit' => ' ',
			'font-size'      => 'var(--o-fluid-font)',
			'font-weight'    => 'var(--o-font-weight-4)',
			'line-height'    => 'var(--o-font-lineheight-3)',
			'color'          => 'var(--o-text-2)',
		);

		$defaults['links'] = array(
			'all'          => array(
				'color'                 => 'var(--o-links)',
				'font-weight'           => '',
				'text-decoration'       => 'none',
				'hover_color'           => 'var(--o-links-hover)',
				'hover_text-decoration' => 'none',
			),
			'text_link'    => array(
				'color'                 => '',
				'font-weight'           => '',
				'text-decoration'       => '',
				'hover_color'           => '',
				'hover_text-decoration' => '',
			),
			'link_wrapper' => array(
				'color'                 => '',
				'font-weight'           => '',
				'text-decoration'       => '',
				'hover_color'           => '',
				'hover_text-decoration' => '',
			),
			'button'       => array(
				'border-radius-unit' => '',
				'font-weight'        => '',
				'border-radius'      => 'var(--o-radius-2)',
			),
		);

		$defaults['sections'] = array(
			'container-padding-top-unit'    => '',
			'container-padding-bottom-unit' => '',
			'container-padding-left-unit'   => '',
			'container-padding-right-unit'  => '',
			'container-padding-top'         => 'var(--o-size-fluid-6)',
			'container-padding-bottom'      => 'var(--o-size-fluid-6)',
			'container-padding-left'        => 'var(--o-size-fluid-4)',
			'container-padding-right'       => 'var(--o-size-fluid-4)',
		);

		$defaults['columns'] = array(
			'padding-top'         => 'var(--o-size-fluid-4)',
			'padding-top-unit'    => '',
			'padding-bottom'      => 'var(--o-size-fluid-4)',
			'padding-bottom-unit' => '',
			'padding-left'        => 'var(--o-size-fluid-4)',
			'padding-left-unit'   => '',
			'padding-right'       => 'var(--o-size-fluid-4)',
			'padding-right-unit'  => '',
		);

		$defaults['aos'] = array(
			'type'             => '',
			'duration'         => '',
			'easing'           => '',
			'offset'           => '',
			'delay'            => '',
			'anchor-placement' => '',
			'once'             => '',
			'mirror'           => '',
			'disable'          => '',
		);

		$defaults['max-width'] = '1440';

		$defaults['breakpoints'] = array(
			'tablet'          => '1024',
			'phone-landscape' => '768',
			'phone-portrait'  => '480',
		);

		return apply_filters( 'oxyprops_global_settings_for_oxygen', $defaults );
	}

	/**
	 * Returns the OxyProps theme for Bricks
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array Gloabal settings for Oxygen.
	 */
	public function get_oxyprops_global_settings_for_bricks() {
		$defaults = array();

		$defaults['label'] = 'OxyProps';

		$defaults['settings']['_custom'] = true;

		$defaults['settings']['general'] = array(
			'siteBackground'    => array(
				'color' => array(
					'raw' => 'var(--o-surface-1)',
				),
			),
			'contentBackground' => array(
				'color' => array(
					'raw' => 'var(--o-surface-1)',
				),
			),
		);

		$defaults['settings']['typography'] = array(
			'typographyHtml'       => '100%',
			'typographyHeadings'   => array(
				'color'       => array(
					'id'   => 'oxyprops-text-1',
					'name' => 'text-1',
					'raw'  => 'var(--o-text-1)',
				),
				'fallback'    => 'var(--o-font-sans)',
				'line-height' => 'var(--o-font-lineheight-1)',
				'font-weight' => '700',
				'font-size'   => 'var(--o-fluid-font)',
				'font-family' => 'var(--o-font-sans)',
			),
			'typographyBody'       => array(
				'color'          => array(
					'id'   => 'oxyprops-text-2',
					'name' => 'text-2',
					'raw'  => 'var(--o-text-2)',
				),
				'line-height'    => 'var(--o-font-lineheight-1)',
				'text-transform' => 'none',
				'font-size'      => 'var(--o-fluid-font)',
				'font-family'    => 'var(--o-font-sans)',
			),
			'blockquoteTypography' => array(
				'font-size' => 'var(--o-fluid-font)',
			),
		);

		$defaults['settings']['conditions'] = array(
			'conditions' =>
			array(
				0 =>
				array(
					'id'   => 'sartnc',
					'main' => 'any',
				),
			),
		);

		$defaults['settings']['links'] = array(
			'typography' =>
			array(
				'color' =>
				array(
					'raw' => 'var(--o-links)',
				),
			),
		);

		$defaults['settings']['section'] = array(
			'padding' =>
			array(
				'left'  => 'var(--o-size-fluid-2)',
				'right' => 'var(--o-size-fluid-2)',
			),
		);

		$defaults['settings']['container'] = array(
			'width'    => '100%',
			'widthMax' => '1360',
		);

		$defaults['settings']['colors'] = array(
			'colorPrimary'   =>
			array(
				'raw' => 'var(--o-brand)',
			),
			'colorSecondary' =>
			array(
				'raw' => 'var(--o-accent)',
			),
			'colorLight'     =>
			array(
				'raw' => 'var(--o-wb)',
			),
			'colorDark'      =>
			array(
				'raw' => 'var(--o-bw)',
			),
			'colorMuted'     =>
			array(
				'id'   => 'oxyprops-surface-4',
				'name' => 'surface-4',
				'raw'  => 'var(--o-surface-4)',
			),
			'colorInfo'      =>
			array(
				'id'   => 'oxyprops-blue-300',
				'name' => 'blue-300',
				'raw'  => 'var(--o-blue-300)',
			),
			'colorSuccess'   =>
			array(
				'id'   => 'oxyprops-green-300',
				'name' => 'green-300',
				'raw'  => 'var(--o-green-300)',
			),
			'colorWarning'   =>
			array(
				'id'   => 'oxyprops-ochre-300',
				'name' => 'ochre-300',
				'raw'  => 'var(--o-ochre-300)',
			),
			'colorDanger'    =>
			array(
				'id'   => 'oxyprops-red-300',
				'name' => 'red-300',
				'raw'  => 'var(--o-red-300)',
			),
		);

		return apply_filters( 'oxyprops_global_settings_for_bricks', $defaults );
	}

	/**
	 * Returns the Original version of Oxygen Global Settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array Factory State Global settings for Oxygen.
	 */
	public function get_original_global_settings_for_oxygen() {
		$defaults = array();

		$defaults['fonts'] = array(
			'Text'    => 'Open Sans',
			'Display' => 'Source Sans Pro',
		);

		$defaults['scripts'] = array(
			'scroll_to_hash'        => '',
			'scroll_to_hash_time'   => '',
			'scroll_to_hash_offset' => '',
		);

		$defaults['indicateParents'] = 'true';

		$defaults['classSuggestionsLimit'] = '5';

		$defaults['headings'] = array(
			'H1' => array(
				'font-size-unit' => 'px',
				'font-size'      => '36',
				'font-weight'    => '700',
				'color'          => '',
			),
			'H2' => array(
				'font-size-unit' => 'px',
				'font-size'      => '30',
				'font-weight'    => '',
				'color'          => '',
			),
			'H3' => array(
				'font-size-unit' => 'px',
				'font-size'      => '24',
				'font-weight'    => '',
				'color'          => '',
			),
			'H4' => array(
				'font-size-unit' => 'px',
				'font-size'      => '20',
				'font-weight'    => '',
				'color'          => '',
			),
			'H5' => array(
				'font-size-unit' => 'px',
				'font-size'      => '18',
				'font-weight'    => '',
				'color'          => '',
			),
			'H6' => array(
				'font-size-unit' => 'px',
				'font-size'      => '16',
				'font-weight'    => '',
				'color'          => '',
			),
		);

		$defaults['body_text'] = array(
			'font-size-unit' => 'px',
			'font-size'      => '16',
			'font-weight'    => '400',
			'line-height'    => '1.6',
			'color'          => '#404040',
		);

		$defaults['links'] = array(
			'all'          => array(
				'color'                 => '#0074db',
				'font-weight'           => '',
				'text-decoration'       => 'none',
				'hover_color'           => '',
				'hover_text-decoration' => 'none',
			),
			'text_link'    => array(
				'color'                 => '',
				'font-weight'           => '',
				'text-decoration'       => '',
				'hover_color'           => '',
				'hover_text-decoration' => '',
			),
			'link_wrapper' => array(
				'color'                 => '',
				'font-weight'           => '',
				'text-decoration'       => '',
				'hover_color'           => '',
				'hover_text-decoration' => '',
			),
			'button'       => array(
				'border-radius-unit' => 'px',
				'font-weight'        => '',
				'border-radius'      => '3',
			),
		);

		$defaults['sections'] = array(
			'container-padding-top-unit'    => 'px',
			'container-padding-bottom-unit' => 'px',
			'container-padding-left-unit'   => 'px',
			'container-padding-right-unit'  => 'px',
			'container-padding-top'         => '75',
			'container-padding-bottom'      => '75',
			'container-padding-left'        => '20',
			'container-padding-right'       => '20',
		);

		$defaults['columns'] = array(
			'padding-top'         => '20',
			'padding-top-unit'    => 'px',
			'padding-bottom'      => '20',
			'padding-bottom-unit' => 'px',
			'padding-left'        => '20',
			'padding-left-unit'   => 'px',
			'padding-right'       => '20',
			'padding-right-unit'  => 'px',
		);

		$defaults['aos'] = array(
			'type'             => '',
			'duration'         => '',
			'easing'           => '',
			'offset'           => '',
			'delay'            => '',
			'anchor-placement' => '',
			'once'             => '',
			'mirror'           => '',
			'disable'          => '',
		);

		$defaults['max-width'] = '1120';

		$defaults['breakpoints'] = array(
			'tablet'          => '992',
			'phone-landscape' => '768',
			'phone-portrait'  => '480',
		);

		return apply_filters( 'oxyprops_original_settings_for_oxygen', $defaults );
	}

	/**
	 * Returns the Plugin Default Settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array Factory State Global settings for Oxygen.
	 */
	public function get_oxyprops_default_settings() {
		$defaults = array();

		$defaults['paletteUI'] = array(
			'showPresets' => true,
		);

		$defaults['palette'] = array(
			array(
				'name'    => 'canvas',
				'hex'     => '#70707A',
				'load'    => true,
				'neutral' => true,
			),
			array(
				'name'    => 'brand',
				'hex'     => '#6366F2',
				'load'    => true,
				'neutral' => false,
			),
			array(
				'name'    => 'accent',
				'hex'     => '#F59F0A',
				'load'    => true,
				'neutral' => false,
			),
			array(
				'name'    => 'links',
				'hex'     => '#0DA2E7',
				'load'    => true,
				'neutral' => false,
			),
			array(
				'name'    => 'visited',
				'hex'     => '#14B8A5',
				'load'    => true,
				'neutral' => false,
			),
		);

		$defaults['shades'] = array(
			array(
				'name'   => '000',
				'LValue' => 100,
			),
			array(
				'name'   => '015',
				'LValue' => 97.5,
			),
			array(
				'name'   => '025',
				'LValue' => 95,
			),
			array(
				'name'   => '050',
				'LValue' => 90,
			),
			array(
				'name'   => '075',
				'LValue' => 80,
			),
			array(
				'name'   => '085',
				'LValue' => 70,
			),
			array(
				'name'   => '100',
				'LValue' => 60,
			),
			array(
				'name'   => '200',
				'LValue' => 55,
			),
			array(
				'name'   => '300',
				'LValue' => 50,
			),
			array(
				'name'   => '400',
				'LValue' => 45,
			),
			array(
				'name'   => '500',
				'LValue' => 40,
			),
			array(
				'name'   => '600',
				'LValue' => 30,
			),
			array(
				'name'   => '700',
				'LValue' => 20,
			),
			array(
				'name'   => '800',
				'LValue' => 15,
			),
			array(
				'name'   => '900',
				'LValue' => 10,
			),
			array(
				'name'   => '1000',
				'LValue' => 0,
			),
		);

		$defaults['features'] = array(
			'context'                              => true,
			'elements'                             => true,
			'o2_reset'                             => false,
			'global_colors'                        => true,
			'normalize'                            => true,
			'stylesheet'                           => false,
			'dequeue_gutenberg'                    => false,
			'css_print'                            => 'file',
			'delete_legacy'                        => false,
			'selectors'                            => 'essentials',
			'editor_styles'                        => true,
			'meta_theme_color'                     => true,
			'meta_theme_color_choice'              => 'brand',
			'meta_theme_color_choice_custom_light' => '',
			'meta_theme_color_choice_custom_dark'  => '',
		);

		$defaults['utilities'] = array(
			'all'           => true,
			'a11y'          => false,
			'animations'    => false,
			'aspect_ratios' => false,
			'borders'       => false,
			'buttons'       => false,
			'cards'         => false,
			'colors'        => false,
			'color_schemes' => false,
			'cursors'       => false,
			'easing'        => false,
			'gradients'     => false,
			'grid12'        => false,
			'layouts'       => false,
			'masks'         => false,
			'position'      => false,
			'scrollbars'    => false,
			'shadows'       => false,
			'sizes'         => false,
			'typography'    => false,
			'zindex'        => false,
		);

		$defaults['theme'] = array(

			'lock_mode'                      => false,
			'load_light'                     => true,
			'load_dark'                      => true,
			'load_dim'                       => false,
			'load_colorful_light'            => false,
			'load_colorful_dark'             => false,

			'brand_light'                    => '500',
			'brand_light_hover'              => '700',
			'accent_light'                   => '100',
			'accent_light_hover'             => '300',
			'link_light'                     => '300',
			'link_light_hover'               => '500',
			'visited_light'                  => '200',
			'visited_light_hover'            => '400',
			'surface1_light'                 => '000',
			'surface2_light'                 => '015',
			'surface3_light'                 => '025',
			'surface4_light'                 => '050',
			'text1_light'                    => '900',
			'text2_light'                    => '700',
			'shadow_color_light'             => '800',
			'shadow_strength_light'          => '2',
			'scrollbar_light'                => '400',

			'brand_dark'                     => '300',
			'brand_dark_hover'               => '100',
			'accent_dark'                    => '085',
			'accent_dark_hover'              => '050',
			'link_dark'                      => '100',
			'link_dark_hover'                => '075',
			'visited_dark'                   => '100',
			'visited_dark_hover'             => '075',
			'surface1_dark'                  => '900',
			'surface2_dark'                  => '800',
			'surface3_dark'                  => '700',
			'surface4_dark'                  => '600',
			'text1_dark'                     => '015',
			'text2_dark'                     => '025',
			'shadow_color_dark'              => '1000',
			'shadow_strength_dark'           => '25',
			'scrollbar_dark'                 => '700',

			'brand_dim'                      => '400',
			'brand_dim_hover'                => '200',
			'accent_dim'                     => '400',
			'accent_dim_hover'               => '200',
			'link_dim'                       => '400',
			'link_dim_hover'                 => '200',
			'visited_dim'                    => '400',
			'visited_dim_hover'              => '200',
			'surface1_dim'                   => '800',
			'surface2_dim'                   => '700',
			'surface3_dim'                   => '600',
			'surface4_dim'                   => '500',
			'text1_dim'                      => '300',
			'text2_dim'                      => '400',
			'shadow_color_dim'               => '900',
			'shadow_strength_dim'            => '30',
			'scrollbar_dim'                  => '600',

			'brand_invert_dim'               => '400',
			'brand_invert_dim_hover'         => '200',
			'accent_invert_dim'              => '400',
			'accent_invert_dim_hover'        => '200',
			'link_invert_dim'                => '400',
			'link_invert_dim_hover'          => '200',
			'visited_invert_dim'             => '400',
			'visited_invert_dim_hover'       => '200',
			'surface1_invert_dim'            => '800',
			'surface2_invert_dim'            => '700',
			'surface3_invert_dim'            => '600',
			'surface4_invert_dim'            => '500',
			'text1_invert_dim'               => '300',
			'text2_invert_dim'               => '400',
			'shadow_color_invert_dim'        => '900',
			'shadow_strength_invert_dim'     => '30',
			'scrollbar_invert_dim'           => '600',

			'brand_colorful_light'           => '500',
			'brand_colorful_light_hover'     => '700',
			'accent_colorful_light'          => '100',
			'accent_colorful_light_hover'    => '300',
			'link_colorful_light'            => '300',
			'link_colorful_light_hover'      => '500',
			'visited_colorful_light'         => '200',
			'visited_colorful_light_hover'   => '400',
			'surface1_colorful_light'        => '015',
			'surface2_colorful_light'        => '025',
			'surface3_colorful_light'        => '050',
			'surface4_colorful_light'        => '075',
			'text1_colorful_light'           => '900',
			'text2_colorful_light'           => '700',
			'shadow_color_colorful_light'    => '800',
			'shadow_strength_colorful_light' => '2',
			'scrollbar_colorful_light'       => '400',

			'brand_colorful_dark'            => '300',
			'brand_colorful_dark_hover'      => '100',
			'accent_colorful_dark'           => '085',
			'accent_colorful_dark_hover'     => '050',
			'link_colorful_dark'             => '100',
			'link_colorful_dark_hover'       => '075',
			'visited_colorful_dark'          => '100',
			'visited_colorful_dark_hover'    => '075',
			'surface1_colorful_dark'         => '900',
			'surface2_colorful_dark'         => '800',
			'surface3_colorful_dark'         => '700',
			'surface4_colorful_dark'         => '600',
			'text1_colorful_dark'            => '015',
			'text2_colorful_dark'            => '025',
			'shadow_color_colorful_dark'     => '1000',
			'shadow_strength_colorful_dark'  => '5',
			'scrollbar_colorful_dark'        => '800',
		);

		$defaults['colors'] = array(
			'canvas'  => array(
				'name'   => 'zinc',
				'hex'    => '#70707a',
				'weight' => '300',
				'swatch' => array(
					100   => array(
						'h' => '240.16',
						's' => '4.9399%',
						'l' => '58.476%',
					),
					200   => array(
						'h' => '240.1',
						's' => '4.3249%',
						'l' => '53.374%',
					),
					300   => array(
						'h' => '240.03',
						's' => '4.1287%',
						'l' => '47.863%',
					),
					400   => array(
						'h' => '240',
						's' => '4.2735%',
						'l' => '45.882%',
					),
					500   => array(
						'h' => '239.87',
						's' => '4.9319%',
						'l' => '38.607%',
					),
					600   => array(
						'h' => '239.63',
						's' => '6.2177%',
						'l' => '29.266%',
					),
					700   => array(
						'h' => '239.25',
						's' => '8.441%',
						'l' => '20.401%',
					),
					800   => array(
						'h' => '238.97',
						's' => '10.305%',
						'l' => '16.173%',
					),
					900   => array(
						'h' => '238.58',
						's' => '13.266%',
						'l' => '12.102%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '240.49',
						's' => '100%',
						'l' => '98.894%',
					),
					'025' => array(
						'h' => '240.41',
						's' => '61.194%',
						'l' => '96.314%',
					),
					'050' => array(
						'h' => '240.39',
						's' => '23.979%',
						'l' => '90.701%',
					),
					'075' => array(
						'h' => '240.33',
						's' => '10.7%',
						'l' => '79.673%',
					),
					'085' => array(
						'h' => '240.25',
						's' => '6.8085%',
						'l' => '68.923%',
					),
				),
				'load'   => true,
			),
			'brand'   => array(
				'name'   => 'indigo',
				'hex'    => '#6366f2',
				'weight' => '300',
				'swatch' => array(
					100   => array(
						'h' => '242.14',
						's' => '100%',
						'l' => '76.884%',
					),
					200   => array(
						'h' => '240.03',
						's' => '100%',
						'l' => '73.589%',
					),
					300   => array(
						'h' => '238.73',
						's' => '84.524%',
						'l' => '67.059%',
					),
					400   => array(
						'h' => '237.48',
						's' => '73.399%',
						'l' => '61.822%',
					),
					500   => array(
						'h' => '235.7',
						's' => '64.225%',
						'l' => '55.792%',
					),
					600   => array(
						'h' => '229.53',
						's' => '72.248%',
						'l' => '42.41%',
					),
					700   => array(
						'h' => '221.49',
						's' => '100%',
						'l' => '26.334%',
					),
					800   => array(
						'h' => '222.31',
						's' => '100%',
						'l' => '21.507%',
					),
					900   => array(
						'h' => '223.67',
						's' => '100%',
						'l' => '16.943%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '250.5',
						's' => '100%',
						'l' => '98.803%',
					),
					'025' => array(
						'h' => '250.19',
						's' => '100%',
						'l' => '97.291%',
					),
					'050' => array(
						'h' => '249.51',
						's' => '100%',
						'l' => '94.29%',
					),
					'075' => array(
						'h' => '247.83',
						's' => '100%',
						'l' => '88.387%',
					),
					'085' => array(
						'h' => '245.53',
						's' => '100%',
						'l' => '82.58%',
					),
				),
				'load'   => true,
			),
			'accent'  => array(
				'name'   => 'gold',
				'hex'    => '#f59f0a',
				'weight' => '300',
				'swatch' => array(
					100   => array(
						'h' => '39.104',
						's' => '100%',
						'l' => '37.455%',
					),
					200   => array(
						'h' => '38.943',
						's' => '100%',
						'l' => '34.19%',
					),
					300   => array(
						'h' => '38.743',
						's' => '100%',
						'l' => '30.629%',
					),
					400   => array(
						'h' => '38.566',
						's' => '100%',
						'l' => '27.713%',
					),
					500   => array(
						'h' => '38.303',
						's' => '100%',
						'l' => '24.56%',
					),
					600   => array(
						'h' => '37.605',
						's' => '100%',
						'l' => '18.53%',
					),
					700   => array(
						'h' => '37.469',
						's' => '100%',
						'l' => '12.649%',
					),
					800   => array(
						'h' => '37.732',
						's' => '100%',
						'l' => '9.8182%',
					),
					900   => array(
						'h' => '38.357',
						's' => '100%',
						'l' => '7.0844%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '30.64',
						's' => '100%',
						'l' => '96.167%',
					),
					'025' => array(
						'h' => '31.091',
						's' => '100%',
						'l' => '92.424%',
					),
					'050' => array(
						'h' => '32.126',
						's' => '100%',
						'l' => '84.27%',
					),
					'075' => array(
						'h' => '36.047',
						's' => '100%',
						'l' => '61.258%',
					),
					'085' => array(
						'h' => '38.043',
						's' => '92.157%',
						'l' => '50%',
					),
				),
				'load'   => true,
			),
			'links'   => array(
				'name'   => 'sky',
				'hex'    => '#0da2e7',
				'weight' => '300',
				'swatch' => array(
					100   => array(
						'h' => '198.99',
						's' => '89.344%',
						'l' => '47.843%',
					),
					200   => array(
						'h' => '197.44',
						's' => '100%',
						'l' => '37.959%',
					),
					300   => array(
						'h' => '197.55',
						's' => '100%',
						'l' => '33.994%',
					),
					400   => array(
						'h' => '197.61',
						's' => '100%',
						'l' => '30.707%',
					),
					500   => array(
						'h' => '197.7',
						's' => '100%',
						'l' => '27.152%',
					),
					600   => array(
						'h' => '198.09',
						's' => '100%',
						'l' => '20.434%',
					),
					700   => array(
						'h' => '198.98',
						's' => '100%',
						'l' => '14.273%',
					),
					800   => array(
						'h' => '199.8',
						's' => '100%',
						'l' => '11.425%',
					),
					900   => array(
						'h' => '201.17',
						's' => '100%',
						'l' => '8.747%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '209.03',
						's' => '100%',
						'l' => '97.666%',
					),
					'025' => array(
						'h' => '208.7',
						's' => '100%',
						'l' => '95.129%',
					),
					'050' => array(
						'h' => '207.9',
						's' => '100%',
						'l' => '89.838%',
					),
					'075' => array(
						'h' => '205.27',
						's' => '100%',
						'l' => '77.282%',
					),
					'085' => array(
						'h' => '201.95',
						's' => '98.03%',
						'l' => '61.461%',
					),
				),
				'load'   => true,
			),
			'visited' => array(
				'name'   => 'teal',
				'hex'    => '#14b8a5',
				'weight' => '300',
				'swatch' => array(
					100   => array(
						'h' => '173.86',
						's' => '100%',
						'l' => '30.794%',
					),
					200   => array(
						'h' => '173.81',
						's' => '100%',
						'l' => '28.07%',
					),
					300   => array(
						'h' => '173.75',
						's' => '100%',
						'l' => '25.073%',
					),
					400   => array(
						'h' => '173.71',
						's' => '100%',
						'l' => '22.572%',
					),
					500   => array(
						'h' => '173.66',
						's' => '100%',
						'l' => '19.868%',
					),
					600   => array(
						'h' => '173.46',
						's' => '100%',
						'l' => '14.706%',
					),
					700   => array(
						'h' => '173.04',
						's' => '100%',
						'l' => '9.9628%',
					),
					800   => array(
						'h' => '172.62',
						's' => '100%',
						'l' => '7.7801%',
					),
					900   => array(
						'h' => '171.83',
						's' => '100%',
						'l' => '5.8383%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '168.24',
						's' => '100%',
						'l' => '90.782%',
					),
					'025' => array(
						'h' => '170.37',
						's' => '100%',
						'l' => '77.591%',
					),
					'050' => array(
						'h' => '170.78',
						's' => '92.227%',
						'l' => '70.713%',
					),
					'075' => array(
						'h' => '171.41',
						's' => '67.117%',
						'l' => '58.603%',
					),
					'085' => array(
						'h' => '173.05',
						's' => '80.392%',
						'l' => '40%',
					),
				),
				'load'   => true,
			),
			'mole'    => array(
				'weight' => '300',
				'name'   => 'mole',
				'hex'    => '#78726d',
				'swatch' => array(
					100   => array(
						'h' => '27.213',
						's' => '5.1738%',
						'l' => '56.456%',
					),
					200   => array(
						'h' => '27.237',
						's' => '4.5517%',
						'l' => '51.386%',
					),
					300   => array(
						'h' => '27.273',
						's' => '4.8035%',
						'l' => '44.902%',
					),
					400   => array(
						'h' => '27.295',
						's' => '5.119%',
						'l' => '41.518%',
					),
					500   => array(
						'h' => '27.332',
						's' => '5.6555%',
						'l' => '36.73%',
					),
					600   => array(
						'h' => '27.428',
						's' => '7.1691%',
						'l' => '27.479%',
					),
					700   => array(
						'h' => '27.574',
						's' => '9.8407%',
						'l' => '18.719%',
					),
					800   => array(
						'h' => '27.681',
						's' => '12.142%',
						'l' => '14.553%',
					),
					900   => array(
						'h' => '27.829',
						's' => '15.94%',
						'l' => '10.552%',
					),
					1000  => array(
						'h' => '26.866',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '26.989',
						's' => '100%',
						'l' => '99.877%',
					),
					'015' => array(
						'h' => '27.101',
						's' => '81.28%',
						'l' => '96.922%',
					),
					'025' => array(
						'h' => '27.106',
						's' => '42.188%',
						'l' => '94.104%',
					),
					'050' => array(
						'h' => '27.117',
						's' => '21.402%',
						'l' => '88.516%',
					),
					'075' => array(
						'h' => '27.143',
						's' => '10.663%',
						'l' => '77.538%',
					),
					'085' => array(
						'h' => '27.174',
						's' => '7.0191%',
						'l' => '66.843%',
					),
				),
				'load'   => true,
			),
			'lead'    => array(
				'weight' => '400',
				'name'   => 'lead',
				'hex'    => '#666e63',
				'swatch' => array(
					100   => array(
						'h' => '103.5',
						's' => '5.1016%',
						'l' => '55.391%',
					),
					200   => array(
						'h' => '103.54',
						's' => '4.5045%',
						'l' => '50.341%',
					),
					300   => array(
						'h' => '103.59',
						's' => '4.8825%',
						'l' => '44.889%',
					),
					400   => array(
						'h' => '103.64',
						's' => '5.2632%',
						'l' => '40.98%',
					),
					500   => array(
						'h' => '103.71',
						's' => '5.8924%',
						'l' => '35.748%',
					),
					600   => array(
						'h' => '103.89',
						's' => '7.5484%',
						'l' => '26.547%',
					),
					700   => array(
						'h' => '104.17',
						's' => '10.55%',
						'l' => '17.847%',
					),
					800   => array(
						'h' => '104.39',
						's' => '13.217%',
						'l' => '13.716%',
					),
					900   => array(
						'h' => '104.04',
						's' => '17.993%',
						'l' => '9.7351%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '103.31',
						's' => '59.107%',
						'l' => '95.739%',
					),
					'025' => array(
						'h' => '103.32',
						's' => '35.411%',
						'l' => '92.928%',
					),
					'050' => array(
						'h' => '103.33',
						's' => '19.573%',
						'l' => '87.354%',
					),
					'075' => array(
						'h' => '103.38',
						's' => '10.232%',
						'l' => '76.406%',
					),
					'085' => array(
						'h' => '103.43',
						's' => '6.8542%',
						'l' => '65.743%',
					),
				),
				'load'   => true,
			),
			'pipe'    => array(
				'weight' => '500',
				'name'   => 'pipe',
				'hex'    => '#616563',
				'swatch' => array(
					100   => array(
						'h' => '149.92',
						's' => '1.9136%',
						'l' => '56.268%',
					),
					200   => array(
						'h' => '149.94',
						's' => '1.6865%',
						'l' => '51.203%',
					),
					300   => array(
						'h' => '149.96',
						's' => '1.7641%',
						'l' => '45.735%',
					),
					400   => array(
						'h' => '149.99',
						's' => '1.9176%',
						'l' => '41.345%',
					),
					500   => array(
						'h' => '150',
						's' => '2.0202%',
						'l' => '38.824%',
					),
					600   => array(
						'h' => '150.09',
						's' => '2.7079%',
						'l' => '27.325%',
					),
					700   => array(
						'h' => '150.22',
						's' => '3.7519%',
						'l' => '18.583%',
					),
					800   => array(
						'h' => '150.31',
						's' => '4.6616%',
						'l' => '14.428%',
					),
					900   => array(
						'h' => '150.44',
						's' => '6.1782%',
						'l' => '10.439%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '149.83',
						's' => '28.044%',
						'l' => '96.705%',
					),
					'025' => array(
						'h' => '149.84',
						's' => '15.037%',
						'l' => '93.889%',
					),
					'050' => array(
						'h' => '149.85',
						's' => '7.7676%',
						'l' => '88.304%',
					),
					'075' => array(
						'h' => '149.87',
						's' => '3.9112%',
						'l' => '77.333%',
					),
					'085' => array(
						'h' => '149.89',
						's' => '2.5867%',
						'l' => '66.646%',
					),
				),
				'load'   => true,
			),
			'denim'   => array(
				'weight' => '400',
				'name'   => 'denim',
				'hex'    => '#537279',
				'swatch' => array(
					100   => array(
						'h' => '191.16',
						's' => '16.761%',
						'l' => '54.155%',
					),
					200   => array(
						'h' => '191.13',
						's' => '15.487%',
						'l' => '49.081%',
					),
					300   => array(
						'h' => '191.09',
						's' => '17.227%',
						'l' => '43.597%',
					),
					400   => array(
						'h' => '191.05',
						's' => '18.627%',
						'l' => '40%',
					),
					500   => array(
						'h' => '190.98',
						's' => '21.415%',
						'l' => '34.373%',
					),
					600   => array(
						'h' => '190.77',
						's' => '28.949%',
						'l' => '25.015%',
					),
					700   => array(
						'h' => '190.28',
						's' => '45.599%',
						'l' => '15.956%',
					),
					800   => array(
						'h' => '189.7',
						's' => '66.121%',
						'l' => '11.386%',
					),
					900   => array(
						'h' => '189.62',
						's' => '100%',
						'l' => '7.3538%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '191.32',
						's' => '100%',
						'l' => '95.268%',
					),
					'025' => array(
						'h' => '191.27',
						's' => '100.05%',
						'l' => '91.768%',
					),
					'050' => array(
						'h' => '191.26',
						's' => '59.092%',
						'l' => '86.188%',
					),
					'075' => array(
						'h' => '191.24',
						's' => '32.324%',
						'l' => '75.226%',
					),
					'085' => array(
						'h' => '191.21',
						's' => '22.134%',
						'l' => '64.541%',
					),
				),
				'load'   => true,
			),
			'grey'    => array(
				'weight' => '400',
				'name'   => 'grey',
				'hex'    => '#646c73',
				'swatch' => array(
					100   => array(
						'h' => '208.14',
						's' => '7.1879%',
						'l' => '57.02%',
					),
					200   => array(
						'h' => '208.1',
						's' => '6.3292%',
						'l' => '51.939%',
					),
					300   => array(
						'h' => '208.05',
						's' => '6.4306%',
						'l' => '46.451%',
					),
					400   => array(
						'h' => '208',
						's' => '6.9767%',
						'l' => '42.157%',
					),
					500   => array(
						'h' => '207.93',
						's' => '7.7467%',
						'l' => '37.241%',
					),
					600   => array(
						'h' => '207.76',
						's' => '9.8915%',
						'l' => '27.954%',
					),
					700   => array(
						'h' => '207.48',
						's' => '13.732%',
						'l' => '19.15%',
					),
					800   => array(
						'h' => '207.26',
						's' => '17.086%',
						'l' => '14.955%',
					),
					900   => array(
						'h' => '206.94',
						's' => '22.685%',
						'l' => '10.916%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '208.32',
						's' => '100%',
						'l' => '97.557%',
					),
					'025' => array(
						'h' => '208.31',
						's' => '64.08%',
						'l' => '94.735%',
					),
					'050' => array(
						'h' => '208.3',
						's' => '30.726%',
						'l' => '89.139%',
					),
					'075' => array(
						'h' => '208.25',
						's' => '14.918%',
						'l' => '78.143%',
					),
					'085' => array(
						'h' => '208.2',
						's' => '9.7585%',
						'l' => '67.429%',
					),
				),
				'load'   => true,
			),
			'slate'   => array(
				'weight' => '500',
				'name'   => 'slate',
				'hex'    => '#5c6470',
				'swatch' => array(
					100   => array(
						'h' => '216.32',
						's' => '9.8907%',
						'l' => '57.946%',
					),
					200   => array(
						'h' => '216.25',
						's' => '8.6889%',
						'l' => '52.845%',
					),
					300   => array(
						'h' => '216.16',
						's' => '8.5043%',
						'l' => '47.335%',
					),
					400   => array(
						'h' => '216.07',
						's' => '9.2383%',
						'l' => '42.909%',
					),
					500   => array(
						'h' => '216',
						's' => '9.8039%',
						'l' => '40%',
					),
					600   => array(
						'h' => '215.64',
						's' => '13.005%',
						'l' => '28.743%',
					),
					700   => array(
						'h' => '215.13',
						's' => '17.95%',
						'l' => '19.871%',
					),
					800   => array(
						'h' => '214.74',
						's' => '22.229%',
						'l' => '15.633%',
					),
					900   => array(
						'h' => '214.17',
						's' => '29.304%',
						'l' => '11.542%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '216.85',
						's' => '100%',
						'l' => '98.182%',
					),
					'025' => array(
						'h' => '216.63',
						's' => '100%',
						'l' => '95.768%',
					),
					'050' => array(
						'h' => '216.6',
						's' => '45.583%',
						'l' => '90.158%',
					),
					'075' => array(
						'h' => '216.53',
						's' => '21.016%',
						'l' => '79.135%',
					),
					'085' => array(
						'h' => '216.44',
						's' => '13.529%',
						'l' => '68.389%',
					),
				),
				'load'   => true,
			),
			'zinc'    => array(
				'weight' => '400',
				'name'   => 'zinc',
				'hex'    => '#70707a',
				'swatch' => array(
					100   => array(
						'h' => '240.16',
						's' => '4.9399%',
						'l' => '58.476%',
					),
					200   => array(
						'h' => '240.1',
						's' => '4.3249%',
						'l' => '53.374%',
					),
					300   => array(
						'h' => '240.03',
						's' => '4.1287%',
						'l' => '47.863%',
					),
					400   => array(
						'h' => '240',
						's' => '4.2735%',
						'l' => '45.882%',
					),
					500   => array(
						'h' => '239.87',
						's' => '4.9319%',
						'l' => '38.607%',
					),
					600   => array(
						'h' => '239.63',
						's' => '6.2177%',
						'l' => '29.266%',
					),
					700   => array(
						'h' => '239.25',
						's' => '8.441%',
						'l' => '20.401%',
					),
					800   => array(
						'h' => '238.97',
						's' => '10.305%',
						'l' => '16.173%',
					),
					900   => array(
						'h' => '238.58',
						's' => '13.266%',
						'l' => '12.102%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '240.49',
						's' => '100%',
						'l' => '98.894%',
					),
					'025' => array(
						'h' => '240.41',
						's' => '61.194%',
						'l' => '96.314%',
					),
					'050' => array(
						'h' => '240.39',
						's' => '23.979%',
						'l' => '90.701%',
					),
					'075' => array(
						'h' => '240.33',
						's' => '10.7%',
						'l' => '79.673%',
					),
					'085' => array(
						'h' => '240.25',
						's' => '6.8085%',
						'l' => '68.923%',
					),
				),
				'load'   => true,
			),
			'steel'   => array(
				'weight' => '400',
				'name'   => 'steel',
				'hex'    => '#73687d',
				'swatch' => array(
					100   => array(
						'h' => '271.62',
						's' => '10.674%',
						'l' => '59.401%',
					),
					200   => array(
						'h' => '271.56',
						's' => '9.3226%',
						'l' => '54.282%',
					),
					300   => array(
						'h' => '271.49',
						's' => '8.573%',
						'l' => '48.75%',
					),
					400   => array(
						'h' => '271.43',
						's' => '9.1703%',
						'l' => '44.902%',
					),
					500   => array(
						'h' => '271.33',
						's' => '10.197%',
						'l' => '39.457%',
					),
					600   => array(
						'h' => '271.09',
						's' => '12.762%',
						'l' => '30.074%',
					),
					700   => array(
						'h' => '270.73',
						's' => '17.118%',
						'l' => '21.159%',
					),
					800   => array(
						'h' => '270.47',
						's' => '20.697%',
						'l' => '16.903%',
					),
					900   => array(
						'h' => '270.13',
						's' => '26.261%',
						'l' => '12.799%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '272.2',
						's' => '100%',
						'l' => '98.57%',
					),
					'025' => array(
						'h' => '271.97',
						's' => '100%',
						'l' => '96.895%',
					),
					'050' => array(
						'h' => '271.86',
						's' => '56.771%',
						'l' => '91.717%',
					),
					'075' => array(
						'h' => '271.79',
						's' => '23.729%',
						'l' => '80.661%',
					),
					'085' => array(
						'h' => '271.72',
						's' => '14.83%',
						'l' => '69.881%',
					),
				),
				'load'   => true,
			),
			'iron'    => array(
				'weight' => '400',
				'name'   => 'iron',
				'hex'    => '#746770',
				'swatch' => array(
					100   => array(
						'h' => '318.66',
						's' => '6.4034%',
						'l' => '57.898%',
					),
					200   => array(
						'h' => '318.6',
						's' => '5.6133%',
						'l' => '52.801%',
					),
					300   => array(
						'h' => '318.53',
						's' => '5.4872%',
						'l' => '47.296%',
					),
					400   => array(
						'h' => '318.46',
						's' => '5.9361%',
						'l' => '42.941%',
					),
					500   => array(
						'h' => '318.37',
						's' => '6.5514%',
						'l' => '38.053%',
					),
					600   => array(
						'h' => '318.14',
						's' => '8.2509%',
						'l' => '28.727%',
					),
					700   => array(
						'h' => '317.78',
						's' => '11.198%',
						'l' => '19.878%',
					),
					800   => array(
						'h' => '317.52',
						's' => '13.683%',
						'l' => '15.659%',
					),
					900   => array(
						'h' => '317.18',
						's' => '17.672%',
						'l' => '11.596%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '319.04',
						's' => '100%',
						'l' => '98.3%',
					),
					'025' => array(
						'h' => '318.93',
						's' => '69.12%',
						'l' => '95.703%',
					),
					'050' => array(
						'h' => '318.9',
						's' => '29.633%',
						'l' => '90.095%',
					),
					'075' => array(
						'h' => '318.84',
						's' => '13.677%',
						'l' => '79.076%',
					),
					'085' => array(
						'h' => '318.76',
						's' => '8.7876%',
						'l' => '68.336%',
					),
				),
				'load'   => true,
			),
			'red'     => array(
				'weight' => '200',
				'name'   => 'red',
				'hex'    => '#ef4343',
				'swatch' => array(
					100   => array(
						'h' => '1.1674',
						's' => '97.698%',
						'l' => '64.717%',
					),
					200   => array(
						'h' => '0',
						's' => '84.314%',
						'l' => '60%',
					),
					300   => array(
						'h' => '357.57',
						's' => '70.878%',
						'l' => '52.047%',
					),
					400   => array(
						'h' => '354.58',
						's' => '76.632%',
						'l' => '45.542%',
					),
					500   => array(
						'h' => '347.72',
						's' => '100%',
						'l' => '35.63%',
					),
					600   => array(
						'h' => '348.58',
						's' => '100%',
						'l' => '27.173%',
					),
					700   => array(
						'h' => '351.31',
						's' => '100%',
						'l' => '19.164%',
					),
					800   => array(
						'h' => '356.35',
						's' => '100%',
						'l' => '15.535%',
					),
					900   => array(
						'h' => '5.995',
						's' => '100%',
						'l' => '11.315%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '8.3411',
						's' => '100%',
						'l' => '98.166%',
					),
					'025' => array(
						'h' => '8.2716',
						's' => '100%',
						'l' => '96.085%',
					),
					'050' => array(
						'h' => '8.0653',
						's' => '100%',
						'l' => '91.892%',
					),
					'075' => array(
						'h' => '7.2147',
						's' => '100%',
						'l' => '83.335%',
					),
					'085' => array(
						'h' => '5.1314',
						's' => '100%',
						'l' => '74.458%',
					),
				),
				'load'   => true,
			),
			'orange'  => array(
				'weight' => '100',
				'name'   => 'orange',
				'hex'    => '#f94706',
				'swatch' => array(
					100   => array(
						'h' => '16.049',
						's' => '95.294%',
						'l' => '50%',
					),
					200   => array(
						'h' => '16.238',
						's' => '100%',
						'l' => '46.411%',
					),
					300   => array(
						'h' => '15.547',
						's' => '100%',
						'l' => '40.953%',
					),
					400   => array(
						'h' => '14.851',
						's' => '100%',
						'l' => '37.32%',
					),
					500   => array(
						'h' => '14.02',
						's' => '100%',
						'l' => '33.351%',
					),
					600   => array(
						'h' => '14.128',
						's' => '100%',
						'l' => '25.136%',
					),
					700   => array(
						'h' => '16.937',
						's' => '100%',
						'l' => '16.96%',
					),
					800   => array(
						'h' => '19.342',
						's' => '100%',
						'l' => '13.002%',
					),
					900   => array(
						'h' => '22.686',
						's' => '100%',
						'l' => '9.1902%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '16.42',
						's' => '100%',
						'l' => '97.706%',
					),
					'025' => array(
						'h' => '16.502',
						's' => '100%',
						'l' => '95.234%',
					),
					'050' => array(
						'h' => '16.637',
						's' => '100%',
						'l' => '90.187%',
					),
					'075' => array(
						'h' => '16.686',
						's' => '100%',
						'l' => '79.532%',
					),
					'085' => array(
						'h' => '16.103',
						's' => '100%',
						'l' => '67.73%',
					),
				),
				'load'   => true,
			),
			'ochre'   => array(
				'weight' => '100',
				'name'   => 'ochre',
				'hex'    => '#f97415',
				'swatch' => array(
					100   => array(
						'h' => '25',
						's' => '95%',
						'l' => '52.941%',
					),
					200   => array(
						'h' => '27.077',
						's' => '100%',
						'l' => '40.359%',
					),
					300   => array(
						'h' => '26.74',
						's' => '100%',
						'l' => '36.25%',
					),
					400   => array(
						'h' => '26.405',
						's' => '100%',
						'l' => '32.918%',
					),
					500   => array(
						'h' => '25.903',
						's' => '100%',
						'l' => '29.334%',
					),
					600   => array(
						'h' => '25.21',
						's' => '100%',
						'l' => '22.226%',
					),
					700   => array(
						'h' => '25.972',
						's' => '100%',
						'l' => '15.111%',
					),
					800   => array(
						'h' => '27.103',
						's' => '100%',
						'l' => '11.649%',
					),
					900   => array(
						'h' => '29.044',
						's' => '100%',
						'l' => '8.3038%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '21.364',
						's' => '100%',
						'l' => '97.308%',
					),
					'025' => array(
						'h' => '21.552',
						's' => '100%',
						'l' => '94.51%',
					),
					'050' => array(
						'h' => '21.933',
						's' => '100%',
						'l' => '88.721%',
					),
					'075' => array(
						'h' => '22.686',
						's' => '100%',
						'l' => '76.055%',
					),
					'085' => array(
						'h' => '24.227',
						's' => '100%',
						'l' => '59.731%',
					),
				),
				'load'   => true,
			),
			'gold'    => array(
				'weight' => '085',
				'name'   => 'gold',
				'hex'    => '#f59f0a',
				'swatch' => array(
					100   => array(
						'h' => '39.104',
						's' => '100%',
						'l' => '37.455%',
					),
					200   => array(
						'h' => '38.943',
						's' => '100%',
						'l' => '34.19%',
					),
					300   => array(
						'h' => '38.743',
						's' => '100%',
						'l' => '30.629%',
					),
					400   => array(
						'h' => '38.566',
						's' => '100%',
						'l' => '27.713%',
					),
					500   => array(
						'h' => '38.303',
						's' => '100%',
						'l' => '24.56%',
					),
					600   => array(
						'h' => '37.605',
						's' => '100%',
						'l' => '18.53%',
					),
					700   => array(
						'h' => '37.469',
						's' => '100%',
						'l' => '12.649%',
					),
					800   => array(
						'h' => '37.732',
						's' => '100%',
						'l' => '9.8182%',
					),
					900   => array(
						'h' => '38.357',
						's' => '100%',
						'l' => '7.0844%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '30.64',
						's' => '100%',
						'l' => '96.167%',
					),
					'025' => array(
						'h' => '31.091',
						's' => '100%',
						'l' => '92.424%',
					),
					'050' => array(
						'h' => '32.126',
						's' => '100%',
						'l' => '84.27%',
					),
					'075' => array(
						'h' => '36.047',
						's' => '100%',
						'l' => '61.258%',
					),
					'085' => array(
						'h' => '38.043',
						's' => '92.157%',
						'l' => '50%',
					),
				),
				'load'   => true,
			),
			'yellow'  => array(
				'weight' => '075',
				'name'   => 'yellow',
				'hex'    => '#e7b008',
				'swatch' => array(
					100   => array(
						'h' => '45.855',
						's' => '100%',
						'l' => '33.977%',
					),
					200   => array(
						'h' => '45.74',
						's' => '100%',
						'l' => '30.996%',
					),
					300   => array(
						'h' => '45.605',
						's' => '100%',
						'l' => '27.728%',
					),
					400   => array(
						'h' => '45.493',
						's' => '100%',
						'l' => '25.035%',
					),
					500   => array(
						'h' => '45.326',
						's' => '100%',
						'l' => '22.122%',
					),
					600   => array(
						'h' => '44.81',
						's' => '100%',
						'l' => '16.595%',
					),
					700   => array(
						'h' => '44.404',
						's' => '100%',
						'l' => '11.333%',
					),
					800   => array(
						'h' => '44.261',
						's' => '100%',
						'l' => '8.8315%',
					),
					900   => array(
						'h' => '44.137',
						's' => '100%',
						'l' => '6.4292%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '37.587',
						's' => '100%',
						'l' => '94.737%',
					),
					'025' => array(
						'h' => '38.372',
						's' => '100%',
						'l' => '89.631%',
					),
					'050' => array(
						'h' => '40.426',
						's' => '100%',
						'l' => '77.145%',
					),
					'075' => array(
						'h' => '45.202',
						's' => '93.305%',
						'l' => '46.863%',
					),
					'085' => array(
						'h' => '46.035',
						's' => '100%',
						'l' => '40.091%',
					),
				),
				'load'   => true,
			),
			'lime'    => array(
				'weight' => '085',
				'name'   => 'lime',
				'hex'    => '#82cb15',
				'swatch' => array(
					100   => array(
						'h' => '82.355',
						's' => '100%',
						'l' => '30.244%',
					),
					200   => array(
						'h' => '82.574',
						's' => '100%',
						'l' => '27.559%',
					),
					300   => array(
						'h' => '82.854',
						's' => '100%',
						'l' => '24.625%',
					),
					400   => array(
						'h' => '83.115',
						's' => '100%',
						'l' => '22.198%',
					),
					500   => array(
						'h' => '83.5',
						's' => '100%',
						'l' => '19.563%',
					),
					600   => array(
						'h' => '84.773',
						's' => '100%',
						'l' => '14.57%',
					),
					700   => array(
						'h' => '84.194',
						's' => '100%',
						'l' => '9.8664%',
					),
					800   => array(
						'h' => '81.822',
						's' => '100%',
						'l' => '7.6205%',
					),
					900   => array(
						'h' => '77.075',
						's' => '100%',
						'l' => '5.4681%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '84.829',
						's' => '100%',
						'l' => '83.13%',
					),
					'025' => array(
						'h' => '85.692',
						's' => '100%',
						'l' => '69.062%',
					),
					'050' => array(
						'h' => '85.632',
						's' => '91.085%',
						'l' => '63.296%',
					),
					'075' => array(
						'h' => '85.099',
						's' => '70.14%',
						'l' => '51.312%',
					),
					'085' => array(
						'h' => '84.066',
						's' => '81.25%',
						'l' => '43.922%',
					),
				),
				'load'   => true,
			),
			'green'   => array(
				'weight' => '085',
				'name'   => 'green',
				'hex'    => '#21c45d',
				'swatch' => array(
					100   => array(
						'h' => '147.69',
						's' => '100%',
						'l' => '31.574%',
					),
					200   => array(
						'h' => '147.44',
						's' => '100%',
						'l' => '28.77%',
					),
					300   => array(
						'h' => '147.12',
						's' => '100%',
						'l' => '25.714%',
					),
					400   => array(
						'h' => '146.87',
						's' => '100%',
						'l' => '23.193%',
					),
					500   => array(
						'h' => '146.52',
						's' => '100%',
						'l' => '20.447%',
					),
					600   => array(
						'h' => '145.32',
						's' => '100%',
						'l' => '15.22%',
					),
					700   => array(
						'h' => '142.83',
						's' => '100%',
						'l' => '10.373%',
					),
					800   => array(
						'h' => '139.26',
						's' => '100%',
						'l' => '8.1318%',
					),
					900   => array(
						'h' => '129.06',
						's' => '100%',
						'l' => '6.124%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '127.31',
						's' => '100%',
						'l' => '93.207%',
					),
					'025' => array(
						'h' => '130.16',
						's' => '100%',
						'l' => '85.055%',
					),
					'050' => array(
						'h' => '135.22',
						's' => '98.876%',
						'l' => '71.213%',
					),
					'075' => array(
						'h' => '137.47',
						's' => '71.286%',
						'l' => '58.978%',
					),
					'085' => array(
						'h' => '142.09',
						's' => '71.179%',
						'l' => '44.902%',
					),
				),
				'load'   => true,
			),
			'jade'    => array(
				'weight' => '085',
				'name'   => 'jade',
				'hex'    => '#10b77f',
				'swatch' => array(
					100   => array(
						'h' => '161.8',
						's' => '100%',
						'l' => '31.219%',
					),
					200   => array(
						'h' => '161.65',
						's' => '100%',
						'l' => '28.453%',
					),
					300   => array(
						'h' => '161.47',
						's' => '100%',
						'l' => '25.424%',
					),
					400   => array(
						'h' => '161.34',
						's' => '100%',
						'l' => '22.913%',
					),
					500   => array(
						'h' => '161.15',
						's' => '100%',
						'l' => '20.187%',
					),
					600   => array(
						'h' => '160.52',
						's' => '100%',
						'l' => '14.989%',
					),
					700   => array(
						'h' => '159.15',
						's' => '100%',
						'l' => '10.199%',
					),
					800   => array(
						'h' => '157.85',
						's' => '100%',
						'l' => '7.9857%',
					),
					900   => array(
						'h' => '155.93',
						's' => '100%',
						'l' => '5.9889%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '148.83',
						's' => '100%',
						'l' => '92.181%',
					),
					'025' => array(
						'h' => '151.52',
						's' => '100%',
						'l' => '82.322%',
					),
					'050' => array(
						'h' => '153.95',
						's' => '95.437%',
						'l' => '71.241%',
					),
					'075' => array(
						'h' => '155.32',
						's' => '68.842%',
						'l' => '59.214%',
					),
					'085' => array(
						'h' => '159.88',
						's' => '83.92%',
						'l' => '39.02%',
					),
				),
				'load'   => true,
			),
			'teal'    => array(
				'weight' => '085',
				'name'   => 'teal',
				'hex'    => '#14b8a5',
				'swatch' => array(
					100   => array(
						'h' => '173.86',
						's' => '100%',
						'l' => '30.794%',
					),
					200   => array(
						'h' => '173.81',
						's' => '100%',
						'l' => '28.07%',
					),
					300   => array(
						'h' => '173.75',
						's' => '100%',
						'l' => '25.073%',
					),
					400   => array(
						'h' => '173.71',
						's' => '100%',
						'l' => '22.572%',
					),
					500   => array(
						'h' => '173.66',
						's' => '100%',
						'l' => '19.868%',
					),
					600   => array(
						'h' => '173.46',
						's' => '100%',
						'l' => '14.706%',
					),
					700   => array(
						'h' => '173.04',
						's' => '100%',
						'l' => '9.9628%',
					),
					800   => array(
						'h' => '172.62',
						's' => '100%',
						'l' => '7.7801%',
					),
					900   => array(
						'h' => '171.83',
						's' => '100%',
						'l' => '5.8383%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '168.24',
						's' => '100%',
						'l' => '90.782%',
					),
					'025' => array(
						'h' => '170.37',
						's' => '100%',
						'l' => '77.591%',
					),
					'050' => array(
						'h' => '170.78',
						's' => '92.227%',
						'l' => '70.713%',
					),
					'075' => array(
						'h' => '171.41',
						's' => '67.117%',
						'l' => '58.603%',
					),
					'085' => array(
						'h' => '173.05',
						's' => '80.392%',
						'l' => '40%',
					),
				),
				'load'   => true,
			),
			'cyan'    => array(
				'weight' => '085',
				'name'   => 'cyan',
				'hex'    => '#07b6d5',
				'swatch' => array(
					100   => array(
						'h' => '188.54',
						's' => '100%',
						'l' => '35.153%',
					),
					200   => array(
						'h' => '188.6',
						's' => '100%',
						'l' => '32.064%',
					),
					300   => array(
						'h' => '188.67',
						's' => '100%',
						'l' => '28.678%',
					),
					400   => array(
						'h' => '188.71',
						's' => '100%',
						'l' => '25.851%',
					),
					500   => array(
						'h' => '188.78',
						's' => '100%',
						'l' => '22.782%',
					),
					600   => array(
						'h' => '189.02',
						's' => '100%',
						'l' => '16.931%',
					),
					700   => array(
						'h' => '189.57',
						's' => '100%',
						'l' => '11.572%',
					),
					800   => array(
						'h' => '190.11',
						's' => '100%',
						'l' => '9.1272%',
					),
					900   => array(
						'h' => '190.99',
						's' => '100%',
						'l' => '6.9187%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '193.49',
						's' => '100%',
						'l' => '95.725%',
					),
					'025' => array(
						'h' => '193.31',
						's' => '100%',
						'l' => '91.517%',
					),
					'050' => array(
						'h' => '192.61',
						's' => '100%',
						'l' => '81.583%',
					),
					'075' => array(
						'h' => '191.13',
						's' => '92.372%',
						'l' => '63.709%',
					),
					'085' => array(
						'h' => '189.03',
						's' => '93.636%',
						'l' => '43.137%',
					),
				),
				'load'   => true,
			),
			'sky'     => array(
				'weight' => '100',
				'name'   => 'sky',
				'hex'    => '#0da2e7',
				'swatch' => array(
					100   => array(
						'h' => '198.99',
						's' => '89.344%',
						'l' => '47.843%',
					),
					200   => array(
						'h' => '197.44',
						's' => '100%',
						'l' => '37.959%',
					),
					300   => array(
						'h' => '197.55',
						's' => '100%',
						'l' => '33.994%',
					),
					400   => array(
						'h' => '197.61',
						's' => '100%',
						'l' => '30.707%',
					),
					500   => array(
						'h' => '197.7',
						's' => '100%',
						'l' => '27.152%',
					),
					600   => array(
						'h' => '198.09',
						's' => '100%',
						'l' => '20.434%',
					),
					700   => array(
						'h' => '198.98',
						's' => '100%',
						'l' => '14.273%',
					),
					800   => array(
						'h' => '199.8',
						's' => '100%',
						'l' => '11.425%',
					),
					900   => array(
						'h' => '201.17',
						's' => '100%',
						'l' => '8.747%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '209.03',
						's' => '100%',
						'l' => '97.666%',
					),
					'025' => array(
						'h' => '208.7',
						's' => '100%',
						'l' => '95.129%',
					),
					'050' => array(
						'h' => '207.9',
						's' => '100%',
						'l' => '89.838%',
					),
					'075' => array(
						'h' => '205.27',
						's' => '100%',
						'l' => '77.282%',
					),
					'085' => array(
						'h' => '201.95',
						's' => '98.03%',
						'l' => '61.461%',
					),
				),
				'load'   => true,
			),
			'blue'    => array(
				'weight' => '200',
				'name'   => 'blue',
				'hex'    => '#3c83f6',
				'swatch' => array(
					100   => array(
						'h' => '218.62',
						's' => '100%',
						'l' => '66.693%',
					),
					200   => array(
						'h' => '217.1',
						's' => '91.176%',
						'l' => '60%',
					),
					300   => array(
						'h' => '214.52',
						's' => '79.821%',
						'l' => '51.817%',
					),
					400   => array(
						'h' => '209.2',
						's' => '100%',
						'l' => '40.494%',
					),
					500   => array(
						'h' => '208.19',
						's' => '100%',
						'l' => '34.349%',
					),
					600   => array(
						'h' => '208.5',
						's' => '100%',
						'l' => '25.994%',
					),
					700   => array(
						'h' => '209.56',
						's' => '100%',
						'l' => '18.586%',
					),
					800   => array(
						'h' => '210.68',
						's' => '100%',
						'l' => '15.242%',
					),
					900   => array(
						'h' => '212.51',
						's' => '100%',
						'l' => '12.088%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '229.78',
						's' => '100%',
						'l' => '98.677%',
					),
					'025' => array(
						'h' => '229.4',
						's' => '100%',
						'l' => '96.979%',
					),
					'050' => array(
						'h' => '228.57',
						's' => '100%',
						'l' => '93.519%',
					),
					'075' => array(
						'h' => '226.45',
						's' => '100%',
						'l' => '86.28%',
					),
					'085' => array(
						'h' => '223.26',
						's' => '100%',
						'l' => '78.065%',
					),
				),
				'load'   => true,
			),
			'indigo'  => array(
				'weight' => '300',
				'name'   => 'indigo',
				'hex'    => '#6467f2',
				'swatch' => array(
					100   => array(
						'h' => '242.14',
						's' => '100%',
						'l' => '76.884%',
					),
					200   => array(
						'h' => '240.03',
						's' => '100%',
						'l' => '73.589%',
					),
					300   => array(
						'h' => '238.73',
						's' => '84.524%',
						'l' => '67.059%',
					),
					400   => array(
						'h' => '237.48',
						's' => '73.399%',
						'l' => '61.822%',
					),
					500   => array(
						'h' => '235.7',
						's' => '64.225%',
						'l' => '55.792%',
					),
					600   => array(
						'h' => '229.53',
						's' => '72.248%',
						'l' => '42.41%',
					),
					700   => array(
						'h' => '221.49',
						's' => '100%',
						'l' => '26.334%',
					),
					800   => array(
						'h' => '222.31',
						's' => '100%',
						'l' => '21.507%',
					),
					900   => array(
						'h' => '223.67',
						's' => '100%',
						'l' => '16.943%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '250.5',
						's' => '100%',
						'l' => '98.803%',
					),
					'025' => array(
						'h' => '250.19',
						's' => '100%',
						'l' => '97.291%',
					),
					'050' => array(
						'h' => '249.51',
						's' => '100%',
						'l' => '94.29%',
					),
					'075' => array(
						'h' => '247.83',
						's' => '100%',
						'l' => '88.387%',
					),
					'085' => array(
						'h' => '245.53',
						's' => '100%',
						'l' => '82.58%',
					),
				),
				'load'   => true,
			),
			'violet'  => array(
				'weight' => '200',
				'name'   => 'violet',
				'hex'    => '#9064f7',
				'swatch' => array(
					100   => array(
						'h' => '258.82',
						's' => '100%',
						'l' => '74.787%',
					),
					200   => array(
						'h' => '257.96',
						's' => '90.184%',
						'l' => '68.039%',
					),
					300   => array(
						'h' => '257.63',
						's' => '80.156%',
						'l' => '64.372%',
					),
					400   => array(
						'h' => '257.15',
						's' => '69.818%',
						'l' => '59.602%',
					),
					500   => array(
						'h' => '256.52',
						's' => '60.937%',
						'l' => '54.361%',
					),
					600   => array(
						'h' => '254.84',
						's' => '61.43%',
						'l' => '44.037%',
					),
					700   => array(
						'h' => '252.17',
						's' => '78.856%',
						'l' => '33.698%',
					),
					800   => array(
						'h' => '249.75',
						's' => '94.612%',
						'l' => '28.245%',
					),
					900   => array(
						'h' => '239.5',
						's' => '100%',
						'l' => '23.625%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '264.09',
						's' => '100%',
						'l' => '98.661%',
					),
					'025' => array(
						'h' => '263.85',
						's' => '100%',
						'l' => '97.052%',
					),
					'050' => array(
						'h' => '263.34',
						's' => '100%',
						'l' => '93.856%',
					),
					'075' => array(
						'h' => '262.14',
						's' => '100%',
						'l' => '87.506%',
					),
					'085' => array(
						'h' => '260.67',
						's' => '100%',
						'l' => '81.158%',
					),
				),
				'load'   => true,
			),
			'purple'  => array(
				'weight' => '200',
				'name'   => 'purple',
				'hex'    => '#a855f7',
				'swatch' => array(
					100   => array(
						'h' => '271.11',
						's' => '100%',
						'l' => '72.486%',
					),
					200   => array(
						'h' => '270.74',
						's' => '91.011%',
						'l' => '65.098%',
					),
					300   => array(
						'h' => '270.63',
						's' => '82.024%',
						'l' => '61.466%',
					),
					400   => array(
						'h' => '270.49',
						's' => '72.277%',
						'l' => '56.515%',
					),
					500   => array(
						'h' => '270.38',
						's' => '63.898%',
						'l' => '50.981%',
					),
					600   => array(
						'h' => '270.88',
						's' => '81.758%',
						'l' => '39.177%',
					),
					700   => array(
						'h' => '272.03',
						's' => '100%',
						'l' => '26.494%',
					),
					800   => array(
						'h' => '271.02',
						's' => '100%',
						'l' => '21.947%',
					),
					900   => array(
						'h' => '269.52',
						's' => '100%',
						'l' => '17.587%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '275.22',
						's' => '100%',
						'l' => '98.534%',
					),
					'025' => array(
						'h' => '274.99',
						's' => '100%',
						'l' => '96.834%',
					),
					'050' => array(
						'h' => '274.49',
						's' => '100%',
						'l' => '93.447%',
					),
					'075' => array(
						'h' => '273.41',
						's' => '100%',
						'l' => '86.647%',
					),
					'085' => array(
						'h' => '272.26',
						's' => '100%',
						'l' => '79.713%',
					),
				),
				'load'   => true,
			),
			'grape'   => array(
				'weight' => '200',
				'name'   => 'grape',
				'hex'    => '#d948ef',
				'swatch' => array(
					100   => array(
						'h' => '292.15',
						's' => '90.871%',
						'l' => '64.08%',
					),
					200   => array(
						'h' => '292.1',
						's' => '83.92%',
						'l' => '60.98%',
					),
					300   => array(
						'h' => '292.03',
						's' => '68.828%',
						'l' => '51.251%',
					),
					400   => array(
						'h' => '292.19',
						's' => '77.763%',
						'l' => '44.879%',
					),
					500   => array(
						'h' => '292.94',
						's' => '100%',
						'l' => '34.791%',
					),
					600   => array(
						'h' => '292.77',
						's' => '100%',
						'l' => '26.435%',
					),
					700   => array(
						'h' => '292.37',
						's' => '100%',
						'l' => '18.848%',
					),
					800   => array(
						'h' => '292.01',
						's' => '100%',
						'l' => '15.36%',
					),
					900   => array(
						'h' => '291.47',
						's' => '100%',
						'l' => '12.051%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '297.17',
						's' => '100%',
						'l' => '98.248%',
					),
					'025' => array(
						'h' => '296.72',
						's' => '100%',
						'l' => '96.326%',
					),
					'050' => array(
						'h' => '295.83',
						's' => '100%',
						'l' => '92.457%',
					),
					'075' => array(
						'h' => '294.16',
						's' => '100%',
						'l' => '84.379%',
					),
					'085' => array(
						'h' => '292.71',
						's' => '100%',
						'l' => '75.38%',
					),
				),
				'load'   => true,
			),
			'pink'    => array(
				'weight' => '200',
				'name'   => 'pink',
				'hex'    => '#ec4699',
				'swatch' => array(
					100   => array(
						'h' => '330.5',
						's' => '89.274%',
						'l' => '63.592%',
					),
					200   => array(
						'h' => '330',
						's' => '81.373%',
						'l' => '60%',
					),
					300   => array(
						'h' => '328.22',
						's' => '66.782%',
						'l' => '50.315%',
					),
					400   => array(
						'h' => '326.36',
						's' => '79.258%',
						'l' => '43.597%',
					),
					500   => array(
						'h' => '322.95',
						's' => '100%',
						'l' => '33.88%',
					),
					600   => array(
						'h' => '323.25',
						's' => '100%',
						'l' => '25.668%',
					),
					700   => array(
						'h' => '324.04',
						's' => '100%',
						'l' => '18.156%',
					),
					800   => array(
						'h' => '324.84',
						's' => '100%',
						'l' => '14.709%',
					),
					900   => array(
						'h' => '326.16',
						's' => '100%',
						'l' => '11.469%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '338.97',
						's' => '100%',
						'l' => '98.394%',
					),
					'025' => array(
						'h' => '338.62',
						's' => '100%',
						'l' => '96.54%',
					),
					'050' => array(
						'h' => '337.83',
						's' => '100%',
						'l' => '92.77%',
					),
					'075' => array(
						'h' => '335.84',
						's' => '100%',
						'l' => '84.821%',
					),
					'085' => array(
						'h' => '332.88',
						's' => '100%',
						'l' => '75.495%',
					),
				),
				'load'   => true,
			),
			'rose'    => array(
				'weight' => '200',
				'name'   => 'rose',
				'hex'    => '#f43e5c',
				'swatch' => array(
					100   => array(
						'h' => '351.2',
						's' => '100%',
						'l' => '64.7%',
					),
					200   => array(
						'h' => '350.11',
						's' => '89.216%',
						'l' => '60%',
					),
					300   => array(
						'h' => '347.06',
						's' => '74.089%',
						'l' => '50.617%',
					),
					400   => array(
						'h' => '342.45',
						's' => '93.497%',
						'l' => '42.191%',
					),
					500   => array(
						'h' => '340.22',
						's' => '100%',
						'l' => '35.335%',
					),
					600   => array(
						'h' => '340.92',
						's' => '100%',
						'l' => '26.929%',
					),
					700   => array(
						'h' => '342.33',
						's' => '100%',
						'l' => '19.1%',
					),
					800   => array(
						'h' => '343.81',
						's' => '100%',
						'l' => '15.414%',
					),
					900   => array(
						'h' => '349.9',
						's' => '100%',
						'l' => '11.973%',
					),
					1000  => array(
						'h' => '0',
						's' => '0%',
						'l' => '0%',
					),
					'000' => array(
						'h' => '0',
						's' => '100%',
						'l' => '100%',
					),
					'015' => array(
						'h' => '1.3266',
						's' => '100%',
						'l' => '98.439%',
					),
					'025' => array(
						'h' => '1.1343',
						's' => '100%',
						'l' => '96.608%',
					),
					'050' => array(
						'h' => '0.6559',
						's' => '100%',
						'l' => '92.945%',
					),
					'075' => array(
						'h' => '359.13',
						's' => '100%',
						'l' => '85.37%',
					),
					'085' => array(
						'h' => '356.22',
						's' => '100%',
						'l' => '76.57%',
					),
				),
				'load'   => true,
			),
		);

		$defaults['typography'] = array(
			'base_size'        => '1',
			'minimum_size'     => '0.8',
			'scale_ratio'      => '1.33',
			'mobile_ratio'     => '1.78',
			'subheading_ratio' => '0.7',
		);

		return apply_filters( 'oxyprops_default_settings', $defaults );
	}

	/**
	 * Returns the Plugin Default Settings
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_default_shades() {
		$defaults = array(
			'000'  => '000',
			'015'  => '015',
			'025'  => '025',
			'050'  => '050',
			'075'  => '075',
			'085'  => '085',
			'100'  => '100',
			'200'  => '200',
			'300'  => '300',
			'400'  => '400',
			'500'  => '500',
			'600'  => '600',
			'700'  => '700',
			'800'  => '800',
			'900'  => '900',
			'1000' => '1000',
		);
		return apply_filters( 'oxyprops_default_shades', $defaults );
	}

	/**
	 * Returns the Plugin Default Colors
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_default_colors() {
		$defaults = array(
			'mole',
			'lead',
			'pipe',
			'denim',
			'grey',
			'slate',
			'steel',
			'iron',
			'zinc',
			'red',
			'rose',
			'pink',
			'grape',
			'purple',
			'violet',
			'indigo',
			'blue',
			'sky',
			'cyan',
			'teal',
			'jade',
			'green',
			'lime',
			'yellow',
			'gold',
			'ochre',
			'orange',
		);
		return apply_filters( 'oxyprops_default_colors', $defaults );
	}

	/**
	 * Returns the Plugin Default Logical Colors
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_logical_colors() {
		$defaults = array(
			'brand',
			'brand-hover',
			'accent',
			'accent-hover',
			'links',
			'links-hover',
			'visited',
			'visited-hover',
			'text-1',
			'text-2',
			'surface-1',
			'surface-2',
			'surface-3',
			'surface-4',
			'bw',
			'wb',
		);
		return apply_filters( 'oxyprops_logical_colors', $defaults );
	}

	/**
	 * Returns the User Custom Colors
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_custom_colors() {
		$defaults = array(
			'canvas',
			'brand',
			'accent',
			'links',
			'visited',
		);
		return apply_filters( 'oxyprops_custom_colors', $defaults );
	}

	/**
	 * Returns the User Custom Colors
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_chromatic_colors() {
		$defaults = array(
			'brand',
			'accent',
			'link',
			'visited',
		);
		return apply_filters( 'oxyprops_chromatic_colors', $defaults );
	}
	/**
	 * Returns available color modes
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_color_modes() {
		$defaults = array(
			'light',
			'dark',
			'dim',
			'dim-invert',
			'colorful-light',
			'colorful-dark',
		);
		return apply_filters( 'oxyprops_chromatic_colors', $defaults );
	}

	/**
	 * Returns the Color Palettes
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_oxyprops_palettes() {
		$defaults = array(
			'OP Logical Colors' => 'logical',
			'OP Custom Palette' => 'custom',
			'OP Color Palette'  => 'color',
		);
		return apply_filters( 'oxyprops_palettes', $defaults );
	}

	/**
	 * Returns Bricks Custom Elements
	 *
	 * @author   Cédric Bontems <cedric@thewebforge.dev>
	 * @since    1.4.0
	 *
	 * @return array
	 */
	public function get_bricks_elements() {
		$bricks_elements = array(
			'light_dark_toggle' => array(
				'title'        => 'Light / Dark Toggle',
				'file_name'    => 'class-o-brx-light-dark-toggle',
				'element_name' => 'bricksprops-light-dark-toggle',
				'stylesheet'   => 'olightdarktoggle',
				'docs_slug'    => 'light-dark-toggle',
			),
			'schemes_switcher'  => array(
				'title'        => 'Schemes Switcher',
				'file_name'    => 'class-o-brx-schemes-switcher',
				'element_name' => 'oschemesswitcher',
				'stylesheet'   => 'oschemesswitcher',
				'docs_slug'    => 'schemes-switcher',
			),
			'menubar'           => array(
				'title'        => 'Menu Bar',
				'file_name'    => 'class-o-brx-menubar',
				'element_name' => 'omenubar',
				'stylesheet'   => 'omenubar',
				'docs_slug'    => 'menubar',
			),
			'menuburger'        => array(
				'title'        => 'Menu Burger',
				'file_name'    => 'class-o-brx-menuburger',
				'element_name' => 'omenuburger',
				'stylesheet'   => 'omenuburger',
				'docs_slug'    => 'menuburger',
			),
			'menubutton'        => array(
				'title'        => 'Menu Button',
				'file_name'    => 'class-o-brx-menubutton',
				'element_name' => 'omenubutton',
				'stylesheet'   => 'omenubutton',
				'docs_slug'    => 'menubutton',
			),
			'faq'               => array(
				'title'        => 'FAQ',
				'file_name'    => 'class-o-brx-faq',
				'element_name' => 'ofaq',
				'stylesheet'   => 'ofaq',
				'docs_slug'    => 'faq',
			),
			'icon'              => array(
				'title'        => 'Unlimited Icon',
				'file_name'    => 'class-o-brx-icon',
				'element_name' => 'oicon',
				'stylesheet'   => 'oicon',
				'docs_slug'    => 'icon',
			),
		);
		return apply_filters( 'oxyprops_bricks_elements', $bricks_elements );
	}
}
